/**
* \file: PluginManager.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Utility
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2015 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <adit_logging.h>

#include <dlfcn.h>
#include <unistd.h>

#include "PluginManager.h"

LOG_IMPORT_CONTEXT(demo) // TODO needs to be generic like in Factory.cpp

namespace adit { namespace utility {

PluginManager::Library::Library()
{
    Handle = nullptr;
    ExitPoint = nullptr;
    ExitCallback = nullptr;
}

PluginManager::~PluginManager()
{
    unloadAll();
}

bool PluginManager::load(const std::string& inName, const std::string& inEntryPoint,
        PluginEntryPointCallbackFn inEntryPointCallback, const std::string& inExitPoint,
        PluginExitPointCallbackFn inExitPointCallback)
{
    if (inEntryPointCallback == nullptr || inExitPointCallback == nullptr)
        return false;

    LOGD_DEBUG((demo, "load plugin %s", inName.c_str()));

    Library lib;
    lib.Handle = dlopen(inName.c_str(), RTLD_NOW);
    char* error = dlerror();

    if (error != nullptr || lib.Handle == nullptr)
    {
        LOG_ERROR((demo, "dlopen(): load plugin %s failed: %s", inName.c_str(), error));
        return false;
    }

    // require entry point
    auto entryPoint = dlsym(lib.Handle, inEntryPoint.c_str());
    error = dlerror();
    if (error != nullptr || entryPoint == nullptr)
    {
        dlclose(lib.Handle);
        LOG_ERROR((demo, "dlsym(): load plugin %s failed at entry point: %s", inName.c_str(),
                error));
        return false;
    }

    lib.ExitCallback = inExitPointCallback;
    // require exit point
    lib.ExitPoint = dlsym(lib.Handle, inExitPoint.c_str());
    error = dlerror();
    if (error != nullptr || lib.ExitPoint == nullptr)
    {
        dlclose(lib.Handle);
        LOG_ERROR((demo, "dlsym(): load plugin %s failed at exit point: %s", inName.c_str(),
                error));
        return false;
    }

    // call entry point
    if (!inEntryPointCallback(entryPoint))
    {
        dlclose(lib.Handle);
        LOG_ERROR((demo, "load plugin %s failed: entry point return false", inName.c_str()));
        return false;
    }

    lib.Name = inName;
    libraries.push_back(lib);
    return true;
}

void PluginManager::unloadAll()
{
    for (auto library : libraries)
    {
        LOGD_DEBUG((demo, "unload plugin %s", library.Name.c_str()));
        if (library.ExitCallback != nullptr && library.ExitPoint != nullptr)
        {
            library.ExitCallback(library.ExitPoint);
        }

        // close loaded plugins
        dlclose(library.Handle);
        char* error = dlerror();
        if (error != nullptr)
        {
            LOG_ERROR((demo, "dlclose(): unload plugin failed: %s", error));
        }
        else
        {
            LOGD_DEBUG((demo, "plugin unmapped and closed"));
        }
    }
    libraries.clear();
}

} } // namespace adit { namespace utility
